// Main JavaScript for Internext

// Toast notification system
function showToast(message, type = 'success') {
    const toastContainer = document.getElementById('toast-container');
    if (!toastContainer) {
        const container = document.createElement('div');
        container.id = 'toast-container';
        container.className = 'toast-container';
        document.body.appendChild(container);
    }

    const toast = document.createElement('div');
    toast.className = `toast ${type}`;
    toast.innerHTML = `
        ${message}
        <span class="close-btn" onclick="this.parentElement.remove()">&times;</span>
    `;

    document.getElementById('toast-container').appendChild(toast);

    // Trigger animation
    setTimeout(() => toast.classList.add('show'), 10);

    // Auto remove after 3 seconds
    setTimeout(() => {
        toast.classList.remove('show');
        setTimeout(() => toast.remove(), 300);
    }, 3000);
}

// Cart functionality
function addToCart(productId, quantity = 1) {
    fetch('api/add_to_cart.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            product_id: productId,
            quantity: quantity
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateCartCount();
            showToast('Product added to cart!', 'success');
        } else {
            showToast('Failed to add product to cart', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred');
    });
}

// Wishlist functionality
function addToWishlist(productId) {
    fetch('api/add_to_wishlist.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            product_id: productId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateWishlistCount();
            showToast('Product added to wishlist!', 'success');
        } else {
            showToast('Failed to add product to wishlist', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred');
    });
}

function removeFromWishlist(productId) {
    fetch('api/remove_from_wishlist.php', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
        },
        body: JSON.stringify({
            product_id: productId
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateWishlistCount();
            showToast('Product removed from wishlist!', 'success');
            location.reload(); // Refresh to show changes
        } else {
            showToast('Failed to remove product from wishlist', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('An error occurred');
    });
}

// Update cart count in header
function updateCartCount() {
    fetch('api/get_cart_count.php', {
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
    .then(response => response.json())
    .then(data => {
        // Update desktop cart count
        const cartCountElement = document.getElementById('cart-count');
        if (cartCountElement) {
            const cartLink = cartCountElement.closest('a');
            cartCountElement.textContent = data.count;

            if (data.count > 0) {
                cartLink.classList.add('has-items');
            } else {
                cartLink.classList.remove('has-items');
            }
        }

        // Update mobile cart count
        const cartCountMobileElement = document.getElementById('cart-count-mobile');
        if (cartCountMobileElement) {
            const cartLink = cartCountMobileElement.closest('a');
            cartCountMobileElement.textContent = data.count;

            if (data.count > 0) {
                cartLink.classList.add('has-items');
            } else {
                cartLink.classList.remove('has-items');
            }
        }
    })
    .catch(error => console.error('Error updating cart count:', error));
}

// Update wishlist count in header
function updateWishlistCount() {
    fetch('api/get_wishlist_count.php', {
        headers: {
            'Cache-Control': 'no-cache'
        }
    })
    .then(response => response.json())
    .then(data => {
        // Update desktop wishlist count
        const wishlistCountElement = document.getElementById('wishlist-count');
        if (wishlistCountElement) {
            const wishlistLink = wishlistCountElement.closest('a');
            wishlistCountElement.textContent = data.count;

            if (data.count > 0) {
                wishlistLink.classList.add('has-items');
            } else {
                wishlistLink.classList.remove('has-items');
            }
        }

        // Update mobile wishlist count
        const wishlistCountMobileElement = document.getElementById('wishlist-count-mobile');
        if (wishlistCountMobileElement) {
            const wishlistLink = wishlistCountMobileElement.closest('a');
            wishlistCountMobileElement.textContent = data.count;

            if (data.count > 0) {
                wishlistLink.classList.add('has-items');
            } else {
                wishlistLink.classList.remove('has-items');
            }
        }
    })
    .catch(error => console.error('Error updating wishlist count:', error));
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    // Add to cart buttons
    document.querySelectorAll('.add-to-cart').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            addToCart(productId);
        });
    });

    // Add to wishlist buttons
    document.querySelectorAll('.add-to-wishlist').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            addToWishlist(productId);
        });
    });

    // Remove from wishlist buttons
    document.querySelectorAll('.remove-from-wishlist').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            removeFromWishlist(productId);
        });
    });


    // Quick view modal functionality
    document.querySelectorAll('.quick-view').forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            const productId = this.getAttribute('data-product-id');
            // Debounce to prevent multiple clicks
            if (!this.disabled) {
                this.disabled = true;
                openQuickView(productId);
                setTimeout(() => {
                    this.disabled = false;
                }, 1000);
            }
        });
    });

    // Prevent card click when clicking overlay buttons
    document.querySelectorAll('.product-overlay').forEach(overlay => {
        overlay.addEventListener('click', function(e) {
            e.stopPropagation();
        });
    });

    // Dropdown menu toggle for mobile
    document.querySelectorAll('.dropdown-toggle').forEach(toggle => {
        toggle.addEventListener('click', function(e) {
            e.preventDefault();
            const dropdownMenu = this.nextElementSibling;
            if (dropdownMenu && dropdownMenu.classList.contains('dropdown-menu')) {
                // Close other dropdowns
                document.querySelectorAll('.dropdown-menu').forEach(menu => {
                    if (menu !== dropdownMenu) {
                        menu.classList.remove('show');
                    }
                });
                dropdownMenu.classList.toggle('show');
            }
        });
    });

    // Close dropdown when clicking outside
    document.addEventListener('click', function(event) {
        if (!event.target.closest('.dropdown')) {
            document.querySelectorAll('.dropdown-menu').forEach(menu => {
                menu.classList.remove('show');
            });
        }
    });

    // Update counts on page load
    updateCartCount();
    updateWishlistCount();
});

// Quick view functionality
function openQuickView(productId) {
    // Prevent multiple modals
    const existingModal = document.getElementById('quick-view-modal');
    if (existingModal && existingModal.style.display === 'block') {
        return;
    }

    // Show loading state
    const modal = document.getElementById('quick-view-modal');
    const modalContent = document.getElementById('modal-product-content');
    modalContent.innerHTML = '<div class="loading">Loading product details...</div>';
    modal.style.display = 'block';

    fetch(`product.php?id=${productId}&quick_view=1`)
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.text();
        })
        .then(data => {
            // Check if modal is still open
            if (modal.style.display !== 'block') {
                return;
            }

            modalContent.innerHTML = data;

            // Re-initialize event listeners for buttons in modal
            initializeModalButtons();

            // Initialize image switching for modal
            initializeImageSwitching();

            // Close modal when clicking close button
            const closeBtn = modal.querySelector('.close-modal');
            if (closeBtn) {
                closeBtn.onclick = function(e) {
                    e.stopPropagation();
                    closeModal();
                };
            }

            // Close modal on escape key
            const escapeHandler = function(event) {
                if (event.key === 'Escape' && modal.style.display === 'block') {
                    closeModal();
                }
            };
            document.addEventListener('keydown', escapeHandler);

            // Store escape handler for cleanup
            modal._escapeHandler = escapeHandler;

            // Close modal when clicking outside (but not on modal content)
            const outsideClickHandler = function(event) {
                if (event.target === modal && modal.style.display === 'block') {
                    event.stopPropagation();
                    closeModal();
                }
            };
            modal._outsideClickHandler = outsideClickHandler;
            modal.addEventListener('click', outsideClickHandler);
        })
        .catch(error => {
            console.error('Error loading quick view:', error);
            modalContent.innerHTML = '<div class="error">Failed to load product details. Please try again.</div>';
            showToast('Failed to load product details', 'error');
        });
}

// Close modal function
function closeModal() {
    const modal = document.getElementById('quick-view-modal');
    if (modal.style.display === 'none') return; // Already closed

    modal.style.display = 'none';

    // Clean up event listeners
    if (modal._escapeHandler) {
        document.removeEventListener('keydown', modal._escapeHandler);
        delete modal._escapeHandler;
    }
    if (modal._outsideClickHandler) {
        modal.removeEventListener('click', modal._outsideClickHandler);
        delete modal._outsideClickHandler;
    }
}

// Initialize buttons in modal
function initializeModalButtons() {
    // Add to cart buttons in modal
    document.querySelectorAll('#quick-view-modal .add-to-cart').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            addToCart(productId);
            // Close modal after adding to cart
            setTimeout(() => {
                closeModal();
            }, 1000);
        });
    });

    // Add to wishlist buttons in modal
    document.querySelectorAll('#quick-view-modal .add-to-wishlist').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            addToWishlist(productId);
        });
    });

    // Remove from wishlist buttons in modal
    document.querySelectorAll('#quick-view-modal .remove-from-wishlist').forEach(button => {
        button.addEventListener('click', function() {
            const productId = this.getAttribute('data-product-id');
            removeFromWishlist(productId);
        });
    });
}

// Initialize image switching for modal
function initializeImageSwitching() {
    const modal = document.getElementById('quick-view-modal');
    const mainImage = modal.querySelector('#mainImage');

    if (mainImage) {
        modal.querySelectorAll('.thumbnail-images img').forEach(thumb => {
            thumb.addEventListener('click', function() {
                mainImage.src = this.src;
            });
        });
    }
}