<?php
session_start();
require_once '../config/database.php';
require_once '../classes/Product.php';
require_once '../classes/Category.php';
require_once '../includes/functions.php';

if (!isLoggedIn() || !isAdmin()) {
    redirect('../login.php');
}

$database = new Database();
$db = $database->getConnection();

$product = new Product($db);
$category = new Category($db);

$message = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['create'])) {
        $product->name = sanitizeInput($_POST['name']);
        $product->description = sanitizeInput($_POST['description']);
        $product->price = sanitizeInput($_POST['price']);
        $product->category_id = sanitizeInput($_POST['category_id']);
        $product->stock_quantity = sanitizeInput($_POST['stock_quantity']);

        // Handle multiple image uploads
        if (isset($_FILES['images'])) {
            $uploadedImages = uploadImages($_FILES['images']);
            $product->images = $uploadedImages;
        }

        if ($product->create()) {
            $message = 'Product created successfully';
        } else {
            $message = 'Failed to create product';
        }
    } elseif (isset($_POST['update'])) {
        $product->id = sanitizeInput($_POST['id']);
        $product->name = sanitizeInput($_POST['name']);
        $product->description = sanitizeInput($_POST['description']);
        $product->price = sanitizeInput($_POST['price']);
        $product->category_id = sanitizeInput($_POST['category_id']);
        $product->stock_quantity = sanitizeInput($_POST['stock_quantity']);

        // Handle multiple image uploads for update
        // First, get existing images
        $query = "SELECT images FROM products WHERE id = ?";
        $stmt = $db->prepare($query);
        $stmt->bindParam(1, $product->id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);
        $existingImages = json_decode($row['images'], true) ?: [];

        // Remove selected images
        if (isset($_POST['remove_images'])) {
            $toRemove = $_POST['remove_images'];
            $existingImages = array_diff($existingImages, $toRemove);
        }

        if (isset($_FILES['images']) && !empty($_FILES['images']['name'][0])) {
            $uploadedImages = uploadImages($_FILES['images']);
            $product->images = $uploadedImages;
        } else {
            $product->images = array_values($existingImages); // Keep filtered existing images
        }

        if ($product->update()) {
            $message = 'Product updated successfully';
        } else {
            $message = 'Failed to update product';
        }
    } elseif (isset($_POST['delete'])) {
        $product->id = sanitizeInput($_POST['id']);
        if ($product->delete()) {
            $message = 'Product deleted successfully';
        } else {
            $message = 'Failed to delete product';
        }
    }
}

$products = $product->read();
$categories = $category->read();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Product Management - Internext Admin</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
</head>
<body>
    <div class="admin-container">
        <?php include 'sidebar.php'; ?>

        <div class="main-content">
            <header>
                <h1>Product Management</h1>
                <button class="btn btn-primary" onclick="showCreateForm()">Add New Product</button>
            </header>

            <?php if ($message): ?>
                <div class="alert alert-success"><?php echo $message; ?></div>
            <?php endif; ?>

            <div class="table-container">
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Category</th>
                            <th>Price</th>
                            <th>Stock</th>
                            <th>Images</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($row = $products->fetch(PDO::FETCH_ASSOC)): ?>
                            <tr>
                                <td><?php echo $row['id']; ?></td>
                                <td><?php echo $row['name']; ?></td>
                                <td><?php echo $row['category_name']; ?></td>
                                <td><?php echo formatPrice($row['price']); ?></td>
                                <td><?php echo $row['stock_quantity']; ?></td>
                                <td>
                                    <?php
                                    $images = json_decode($row['images'], true);
                                    if ($images && count($images) > 0) {
                                        echo count($images) . ' image(s)';
                                    } else {
                                        echo 'No images';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <button class="btn btn-small" onclick="editProduct(<?php echo $row['id']; ?>)">Edit</button>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="id" value="<?php echo $row['id']; ?>">
                                        <button type="submit" name="delete" class="btn btn-small btn-danger" onclick="return confirm('Are you sure?')">Delete</button>
                                    </form>
                                </td>
                            </tr>
                        <?php endwhile; ?>
                    </tbody>
                </table>
            </div>

            <!-- Create/Edit Product Modal -->
            <div id="productModal" class="modal">
                <div class="modal-content">
                    <span class="close" onclick="closeModal()">&times;</span>
                    <h2 id="modalTitle">Add New Product</h2>
                    <form id="productForm" method="POST" enctype="multipart/form-data">
                        <input type="hidden" name="id" id="productId">
                        <div class="form-group">
                            <label for="name">Name:</label>
                            <input type="text" id="name" name="name" required>
                        </div>
                        <div class="form-group">
                            <label for="description">Description:</label>
                            <textarea id="description" name="description" required></textarea>
                        </div>
                        <div class="form-group">
                            <label for="price">Price:</label>
                            <input type="number" id="price" name="price" step="0.01" required>
                        </div>
                        <div class="form-group">
                            <label for="category_id">Category:</label>
                            <select id="category_id" name="category_id" required>
                                <option value="">Select Category</option>
                                <?php while ($cat = $categories->fetch(PDO::FETCH_ASSOC)): ?>
                                    <option value="<?php echo $cat['id']; ?>"><?php echo $cat['name']; ?></option>
                                <?php endwhile; ?>
                            </select>
                        </div>
                        <div class="form-group">
                            <label for="stock_quantity">Stock Quantity:</label>
                            <input type="number" id="stock_quantity" name="stock_quantity" required>
                        </div>
                        <div class="form-group">
                            <label for="images">Images (Multiple allowed):</label>
                            <div id="imagePreviews" style="margin-bottom: 10px;"></div>
                            <input type="file" id="images" name="images[]" multiple accept="image/*">
                        </div>
                        <button type="submit" name="create" id="submitBtn" class="btn btn-primary">Create Product</button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <script>
        function showCreateForm() {
            document.getElementById('modalTitle').textContent = 'Add New Product';
            document.getElementById('productForm').reset();
            document.getElementById('productId').value = '';
            document.getElementById('imagePreviews').innerHTML = '';
            document.getElementById('submitBtn').name = 'create';
            document.getElementById('productModal').style.display = 'block';
        }

        function editProduct(id) {
            // Fetch product data and populate form
            fetch(`get_product.php?id=${id}`)
                .then(response => response.json())
                .then(data => {
                    document.getElementById('modalTitle').textContent = 'Edit Product';
                    document.getElementById('productId').value = data.id;
                    document.getElementById('name').value = data.name;
                    document.getElementById('description').value = data.description;
                    document.getElementById('price').value = data.price;
                    document.getElementById('category_id').value = data.category_id;
                    document.getElementById('stock_quantity').value = data.stock_quantity;
                    document.getElementById('submitBtn').name = 'update';

                    // Display existing images
                    let previews = '';
                    if (data.images && data.images.length > 0) {
                        data.images.forEach(img => {
                            previews += `<div style="display:inline-block; margin:5px; text-align:center;">
                                <img src="../uploads/${img}" style="width:100px; height:100px; object-fit:cover;"><br>
                                <input type="checkbox" name="remove_images[]" value="${img}"> Remove
                            </div>`;
                        });
                    }
                    document.getElementById('imagePreviews').innerHTML = previews;

                    document.getElementById('productModal').style.display = 'block';
                });
        }

        function closeModal() {
            document.getElementById('productModal').style.display = 'none';
        }
    </script>
</body>
</html>